﻿#include "precompiled.h"
#include "common.h"
#include "RTSimulation.h"

#include "Scene.h"
#include "Entity.h"

// Components
#include "Transform.h"
#include "Camera.h"
#include "MeshRenderer.h"

const float kCameraPosDelta = 0.5f;

// TODO: Write a component manager

namespace RTCam {

RTSimulation::RTSimulation(void)
{
}


RTSimulation::~RTSimulation(void)
{
}

void RTSimulation::RenderUpdate(const shared_ptr<Scene>& scene, float curTime, float delta)
{
	for(auto& e : scene->m_entities) {
		UpdateModelCBufferDatas(e);
	}
}

void RTSimulation::FixedUpdate(const shared_ptr<Scene>& scene, float curTime, float delta)
{
	UpdateCameraAnimations(scene, curTime, delta);
}

void RTSimulation::UpdateModelCBufferDatas(const shared_ptr<Entity>& e)
{
	auto transform = e->GetTransform();
	ASSERT(transform != nullptr);
	
	transform->UpdateWorldAndNormalMatrices();

	auto renderer = e->GetRenderer();
	if(renderer != nullptr) {
		renderer->UpdateCBuffer();
	}
	
	for(auto& child : e->GetChildren()) {
		ASSERT(child != nullptr);
		UpdateModelCBufferDatas(child);
	}
}

void RTSimulation::UpdateCameraAnimations( const shared_ptr<Scene>& scene, float curTime, float delta )
{
	// TODO: Make it update all cameras, not just the active one
	shared_ptr<Camera> cam = scene->m_activeCam.lock();
	if(cam != nullptr) {
		cam->FixedUpdate(curTime, delta);
	}
}

void RTSimulation::MoveCameraForward(const shared_ptr<Camera>& camera)
{
	auto transform = camera->m_entity.lock()->GetTransform();
	XMFLOAT3A position = transform->GetLocalPosition();
	
	XMFLOAT3A forward = transform->GetForward();
	forward.x *= kCameraPosDelta;
	forward.y *= kCameraPosDelta;
	forward.z *= kCameraPosDelta;

	position.x += forward.x;
	position.y += forward.y;
	position.z += forward.z;
	
	transform->SetLocalPosition(position);
	DebugPrint("Camera is now at %f, %f, %f\n", position.x, position.y, position.z);
}

void RTSimulation::MoveCameraBack(const shared_ptr<Camera>& camera)
{
	auto transform = camera->m_entity.lock()->GetTransform();
	XMFLOAT3A position = transform->GetLocalPosition();

	XMFLOAT3A forward = transform->GetForward();
	forward.x *= kCameraPosDelta;
	forward.y *= kCameraPosDelta;
	forward.z *= kCameraPosDelta;

	position.x -= forward.x;
	position.y -= forward.y;
	position.z -= forward.z;

	transform->SetLocalPosition(position);
	DebugPrint("Camera is now at %f, %f, %f\n", position.x, position.y, position.z);
}

void RTSimulation::MoveCameraRight(const shared_ptr<Camera>& camera)
{
	auto transform = camera->m_entity.lock()->GetTransform();
	XMFLOAT3A position = transform->GetLocalPosition();
	
	XMFLOAT3A right = transform->GetRight();
	right.x *= kCameraPosDelta;
	right.y *= kCameraPosDelta;
	right.z *= kCameraPosDelta;

	position.x += right.x;
	position.y += right.y;
	position.z += right.z;

	transform->SetLocalPosition(position);
	DebugPrint("Camera is now at %f, %f, %f\n", position.x, position.y, position.z);
}

void RTSimulation::MoveCameraLeft(const shared_ptr<Camera>& camera)
{
	auto transform = camera->m_entity.lock()->GetTransform();
	XMFLOAT3A position = transform->GetLocalPosition();
	
	XMFLOAT3A right = transform->GetRight();
	right.x *= kCameraPosDelta;
	right.y *= kCameraPosDelta;
	right.z *= kCameraPosDelta;

	position.x -= right.x;
	position.y -= right.y;
	position.z -= right.z;

	transform->SetLocalPosition(position);
	DebugPrint("Camera is now at %f, %f, %f\n", position.x, position.y, position.z);
}

void RTSimulation::MoveCameraUp( const shared_ptr<Camera>& camera )
{
	auto transform = camera->m_entity.lock()->GetTransform();
	XMFLOAT3A position = transform->GetLocalPosition();
	

	XMFLOAT3A up = transform->GetUp();
	up.x *= kCameraPosDelta;
	up.y *= kCameraPosDelta;
	up.z *= kCameraPosDelta;

	position.x += up.x;
	position.y += up.y;
	position.z += up.z;

	transform->SetLocalPosition(position);
	DebugPrint("Camera is now at %f, %f, %f\n", position.x, position.y, position.z);
}

void RTSimulation::MoveCameraDown( const shared_ptr<Camera>& camera )
{
	auto transform = camera->m_entity.lock()->GetTransform();
	XMFLOAT3A position = transform->GetLocalPosition();
	
	XMFLOAT3A up = transform->GetUp();
	up.x *= kCameraPosDelta;
	up.y *= kCameraPosDelta;
	up.z *= kCameraPosDelta;

	position.x -= up.x;
	position.y -= up.y;
	position.z -= up.z;

	transform->SetLocalPosition(position);
	DebugPrint("Camera is now at %f, %f, %f\n", position.x, position.y, position.z);
}

void RTSimulation::TurnCameraLeft( const shared_ptr<Camera>& camera )
{
	using namespace DirectX;
	auto transform = camera->m_entity.lock()->GetTransform();
	
	XMVECTOR curRotation = XMLoadFloat4A(&transform->GetLocalRotation());
	XMVECTOR addRotation = XMQuaternionRotationRollPitchYaw(0, -XM_PIDIV4 / 2, 0);
	XMVECTOR newRotation = XMQuaternionMultiply(curRotation, addRotation);
	
	XMFLOAT4A rotation;
	XMStoreFloat4A(&rotation, newRotation);
	transform->SetLocalRotation(rotation);
}

void RTSimulation::TurnCameraRight( const shared_ptr<Camera>& camera )
{
	using namespace DirectX;
	auto transform = camera->m_entity.lock()->GetTransform();

	XMVECTOR curRotation = XMLoadFloat4A(&transform->GetLocalRotation());
	XMVECTOR addRotation = XMQuaternionRotationRollPitchYaw(0, XM_PIDIV4 / 2, 0);
	XMVECTOR newRotation = XMQuaternionMultiply(curRotation, addRotation);

	XMFLOAT4A rotation;
	XMStoreFloat4A(&rotation, newRotation);
	transform->SetLocalRotation(rotation);

}

} // end namespace